const fs = require('fs');
const path = require('path');
const url = require('url');
const express = require('express');
const multer = require('multer');
const cors = require('cors');
const pewModule = require('./pew_module');
const pew = new pewModule.pewModule();
const app = express();

//Catch exceptions
process.on('uncaughtException', function (err)
{
    try 
    {
        console.log("Uncaught exception: " + err.stack);
        pew.Constants.DebugLog("Uncaught exception: " + err.stack)
    }
    catch (ex) {}
})

//Check if rest api should be started
pew.autoStartApps();

//use cors for internal testing (sending request from a different client)
app.use(cors());

app.use(express.json({ limit: '50mb' }));
//Serve static files
app.use(express.static("public"));

//provide files for download
app.get(`${pew.Constants.requests.GET_FILES}*`, (req, res) =>
{
    var urlparts = req.url.split("/");
    var filename;

    try 
    {
        filename = urlparts[urlparts.length - 1];
        //Debug information
        pew.Constants.DebugLog(`${pew.Constants.SINGLE_FILE_DOWNLOAD_REQ}: ${filename}`);
        var filepath;
        //Provide config (cfg) files for data logger. The URL deep level is 3 for cfg files.
        if (urlparts.length === pew.Constants.URL_LEVEL_CFG_FILES) 
        {            
            //special case for script file -> just provide the content of the file
            if (filename === pew.Constants.SCRIPT_FILE_NAME)
            {
                filepath = path.join(__dirname, pew.Constants.configSubfolders.script, filename)

                fs.exists(filepath, exists =>
                {
                    if (!exists)
                    {
                        return pew.returnWithFailHeaderAndMsg(`${pew.Constants.FILE_NOT_FOUND}: ${filename}`, false, '', res);
                    }

                    fs.readFile(filepath, (err, data) =>
                    {
                        if (err) 
                        {
                            return pew.returnWithFailHeaderAndMsg(`${pew.Constants.READ_FILE_FAILED}: ${err}`, false, '', res);
                        }
                        res.write(data)
                        res.end()
                    })
                })
            }
            //provide files for download (datalogger format files)
            else 
            {
                filepath = decodeURI(path.join(__dirname, pew.Constants.configSubfolders.log, filename));

                if (fs.existsSync(filepath))
                {
                    res.download(filepath)
                }
                else 
                {
                    pew.returnWithFailHeaderAndMsg(`${pew.Constants.FILE_NOT_FOUND}: ${filename}`, false, pew.Constants.FILE_NOT_FOUND, res);
                }
            }
        }

        if (urlparts.length > pew.Constants.URL_LEVEL_CFG_FILES) 
        {
            //Provide log files
            var logStorage = urlparts[urlparts.length - 2];
            filepath = pew.Constants.EMPTY_STRING;

            switch (logStorage) 
            {
                case pew.Constants.INTERNAL_MEMORY_LINK:
                    filepath = filename === pew.Constants.GET_ALL_FILES ? pew.Constants.Storage.INTERN.path : decodeURI(path.join(pew.Constants.Storage.INTERN.path, filename));
                    break;
                case pew.Constants.USB_MEMORY_LINK:
                    filepath = filename === pew.Constants.GET_ALL_FILES ? pew.Constants.Storage.USBSTICK.path : decodeURI(path.join(pew.Constants.Storage.USBSTICK.path, filename));
                    break;
                case pew.Constants.TEMP_MEMORY_LINK:
                    filepath = filename === pew.Constants.GET_ALL_FILES ? pew.Constants.Storage.TEMP.path : decodeURI(path.join(pew.Constants.Storage.TEMP.path, filename));
                    break;

                default:
                    filepath = filename === pew.Constants.GET_ALL_FILES ? pew.Constants.Storage.INTERN.path : decodeURI(path.join(pew.Constants.Storage.INTERN.path, filename));
            }

            if (filename === pew.Constants.GET_ALL_FILES) 
            {
                pew.zipLogFilesAndProvideForDownload(filepath).then(content =>
                {
                    res.write(content)
                    return res.end();
                }).catch(err =>
                {
                    return pew.returnWithFailHeaderAndMsg(false, pew.Constants.httpStatus.BAD_REQUEST, err, res);
                })
            }
            else
            {
                res.download(filepath);
            }
        }
    }
    catch (err)
    {
        pew.returnWithFailHeaderAndMsg(`${pew.Constants.READ_CFG_FAILED}: ${err}`, false, err, res);
    }
})

//Set up the redirects for settings
app.get(`${pew.Constants.requests.READ_INTERFACE}*`, (req, res) =>
{
    var uri = url.parse(req.url).pathname;
    var filename = path.join(__dirname, pew.Constants.configFolder, uri + pew.Constants.JSONFILE)

    fs.readFile(filename, (err, data) =>
    {
        if (err)
        {
            return pew.returnWithFailHeaderAndMsg(false, pew.Constants.httpStatus.NOT_FOUND, pew.Constants.FILE_NOT_FOUND, res);
        }
        res.write(data)
        res.end()
    })
})

//Some standard get requests
app.get(`${pew.Constants.requests.GET}*`, function (req, res) 
{
    var uri = url.parse(req.url).pathname;
    switch (uri)
    {
        //special request for CFG files of data logger (read all cfg files of folder)
        case pew.Constants.requests.GET_FORMATFILES:
            pew.getCFGFiles().then(data =>
            {
                let result = { "files": [] };
                if (!pew.isUndefined(data) && data.length > 0)
                {
                    for (let file of data) 
                    {
                        result.files.push(file);
                    }
                    res.write(JSON.stringify(result));
                    return res.end();
                }
                else
                {
                    pew.returnWithFailHeaderAndMsg(false, pew.Constants.httpStatus.SERVER_ERR, pew.Constants.FAILED, res);
                }
                res.write(JSON.stringify(result));
                res.end();
            }).catch(err =>
            {
                //In error case 'files' content is the error message
                res.write(err);
                return res.end();
            })
            break;

        case pew.Constants.requests.GETVER:
            pew.getVersion().then(data =>
            {
                res.write(data);
                res.end();
            }).catch(() =>
            {
                pew.returnWithFailHeaderAndMsg(false, pew.Constants.httpStatus.SERVER_ERR, pew.Constants.FAILED, res);
            })
            break;

        case pew.Constants.requests.GET_DEFAULT_SCRIPT:
            pew.getDefaultScript().then(data =>
            {
                res.end(data);
            }).catch(() =>
            {
                return pew.returnWithFailHeaderAndMsg(false, pew.Constants.httpStatus.SERVER_ERR, pew.Constants.FAILED, res);
            })
            break;

        case pew.Constants.requests.GETFILELIST_INTERN:
            pew.getListOfFilesInDir(pew.Constants.Storage.INTERN.path).then(data =>
            {
                res.end(data);
            }).catch(err =>
            {
                pew.returnWithFailHeaderAndMsg(false, pew.Constants.httpStatus.SERVER_ERR, `${pew.Constants.FAILED}: ${err}`, res);
            })
            break;

        case pew.Constants.requests.GETFILELIST_USB:
            pew.getListOfFilesInDir(pew.Constants.Storage.USBSTICK.path).then(data =>
            {
                res.end(data);
            }).catch(err =>
            {
                pew.returnWithFailHeaderAndMsg(false, pew.Constants.httpStatus.SERVER_ERR, `${pew.Constants.FAILED}: ${err}`, res);
            })
            break;

        case pew.Constants.requests.GETFILELIST_TEMP:
            pew.getListOfFilesInDir(pew.Constants.Storage.TEMP.path).then(data =>
            {
                res.end(data);
            }).catch(err =>
            {
                pew.returnWithFailHeaderAndMsg(false, pew.Constants.httpStatus.SERVER_ERR, `${pew.Constants.FAILED}: ${err}`, res);
            })
            break;

        case pew.Constants.requests.GET_UPDATE_STATUS:
            pew.updateStatus().then(() =>
            {
                res.end(pew.Constants.UPDATE_SUCCESS);
            }).catch(() =>
            {
                pew.returnWithFailHeaderAndMsg(false, pew.Constants.httpStatus.SERVER_ERR, pew.Constants.FAILED, res);
            })
            break;

        case pew.Constants.requests.RESET:
            pew.resetApplicationSettings(req).then(msg =>
            {
                res.end(msg);
            }).catch(err =>
            {
                pew.returnWithFailHeaderAndMsg(false, pew.Constants.httpStatus.SERVER_ERR, err, res);
            })
            break;

        case pew.Constants.requests.GET_HTTP_CONTENT:
            pew.getFolderContent(path.join(__dirname, pew.Constants.configSubfolders.http_server)).then(files => {
                res.end(files);
            }).catch(err => {
                pew.returnWithFailHeaderAndMsg(false, pew.Constants.httpStatus.SERVER_ERR, err, res);
            })
            break;

        //Get live count values
        case pew.Constants.requests.GET_COUNT_VALUES:
            if (pew.Constants.isCustomized)
            {
                pew.getCountValues().then(result =>
                {
                    res.end(result);
                }).catch(err =>
                {
                    pew.returnWithFailHeaderAndMsg(false, pew.Constants.httpStatus.SERVER_ERR, err, res);
                })
            }
            else
            {
                pew.returnWithFailHeaderAndMsg(false, pew.Constants.httpStatus.BAD_REQUEST, pew.Constants.FAILED, res);
            }
            break;

        //Get error logs
        case pew.Constants.requests.GET_ERR_LOGS:
            if (pew.Constants.isCustomized)
            {
                pew.getErrorLogs().then(result =>
                {
                    res.end(result);
                }).catch(err =>
                {
                    pew.returnWithFailHeaderAndMsg(false, pew.Constants.httpStatus.SERVER_ERR, err, res);
                })
            }
            else
            {
                pew.returnWithFailHeaderAndMsg(false, pew.Constants.httpStatus.BAD_REQUEST, pew.Constants.FAILED, res);
            }
            break;

        case pew.Constants.requests.GET_PLC_BACKUP:
            if (pew.Constants.isCustomized)
            {
                pew.createPLCBackup().then(result =>
                {
                    res.end(result);
                }).catch(err =>
                {
                    pew.returnWithFailHeaderAndMsg(false, pew.Constants.httpStatus.SERVER_ERR, err, res);
                })
            }
            else
            {
                pew.returnWithFailHeaderAndMsg(false, pew.Constants.httpStatus.BAD_REQUEST, pew.Constants.FAILED, res);
            }
            break;

        case pew.Constants.requests.SQLTEST:
            pew.sqlTestConnection().then(response => {
                res.end(response);
            }).catch(err => {
                res.end(err);
            })
            break;

        case pew.Constants.requests.NOSQLTEST:
            pew.nosqlTestConnection().then(response => {
                res.end(response);
            }).catch(err => {
                res.end(err);
            })
            break;

        case pew.Constants.requests.GET_SYS_LOGS:
            pew.getSysLogs().then(response => {
                res.end(response);
            }).catch(err => {
                pew.returnWithFailHeaderAndMsg(false, pew.Constants.httpStatus.SERVER_ERR, err, res);
            })
            break;

        default:
            pew.returnWithFailHeaderAndMsg(false, pew.Constants.httpStatus.BAD_REQUEST, pew.Constants.FAILED, res);
    }
})

app.post(pew.Constants.requests.BACKUP, function (req, res) {
    //Don't allow requests from external
    if (!pew.isRequestFromLocalhost(req.ip)) {
        pew.returnForbidden(req.ip, res);
        return;
    }

    pew.createBackup(req.body.backupPW).then(zipcontent => {
        res.write(zipcontent);
        res.end();
    }).catch(err => {
        return pew.returnWithFailHeaderAndMsg(false, pew.Constants.httpStatus.SERVER_ERR, err, res);
    })
})

//Set up post redirects for change settings. Also send a note to the application that the setting(s) has been changed with kill -s
app.post(`${pew.Constants.requests.READ_INTERFACE}*`, (req, res) =>
{
    //Don't allow requests from external
    if (!pew.isRequestFromLocalhost(req.ip))
    {
        pew.returnForbidden(req.ip, res);
        return;
    }

    var LocalConstants = pew.Constants;
    var uri = url.parse(req.url).pathname;
    var filename = path.join(__dirname, pew.Constants.configFolder, `${uri}${pew.Constants.JSONFILE}`);    

    //Debug information
    LocalConstants.DebugLog(`${LocalConstants.SAVE_REQ}: ${uri}`);
    let configToWrite = pew.encryptIncomingPW(uri, req.body);

    fs.writeFile(filename, configToWrite, err =>
    {
        if (err)
        {
            //Debug information
            return pew.returnWithFailHeaderAndMsg(`${LocalConstants.SAVE_FAILED}: ${uri}`, LocalConstants.httpStatus.BAD_REQUEST, '', res);
        }
        let diff, delaytime, apiPort, secondRequestTime, firstRequestTime = 'undefined';
        switch (uri) {
            case LocalConstants.requests.READ_API:
                //For the api we need to check if it got enabled or disabled to start or end the daemon
                if (typeof (req.body.api_enable) === LocalConstants.UNDEFINED) {
                    return res.end(LocalConstants.FAILED);
                }

                apiPort = req.body.http_port;

                //API enabled
                if (req.body.api_enable || req.body.http_server) {
                    pew.restartProcess(LocalConstants.API_SERVER, `${LocalConstants.NODE_V16_PATH} ${LocalConstants.NODEJS_SERVER_FILES_PATH}${LocalConstants.API_APP_PATH} ${apiPort}`).then(() => {
                        res.end();
                    }).catch(() => {
                        return res.end();
                    });
                }
                //API Not enabled
                else {
                    pew.killProcessByName(LocalConstants.API_SERVER).then(() => {
                        res.end(LocalConstants.SAVE_OK);
                    }).catch(err => {
                        return res.end(err.message);
                    })
                }
                break;

            case LocalConstants.requests.READ_SQL:
                //sql enabled
                if (req.body.enable_sql) {
                    pew.restartProcess(LocalConstants.SQL_CLIENT, `${LocalConstants.NODE_V16_PATH} ${LocalConstants.NODEJS_SERVER_FILES_PATH}${LocalConstants.SQL_APP_PATH}`).then(() => {
                        res.end();
                    }).catch(() => {
                        return res.end();
                    });
                }
                //sql Not enabled
                else {
                    pew.killProcessByName(LocalConstants.SQL_CLIENT).then(() => {
                        res.end(LocalConstants.SAVE_OK);
                    }).catch(err => {
                        return res.end(err.message);
                    })
                }
                break;

            case LocalConstants.requests.READ_NOSQL:
                //nosql enabled
                if (req.body.enable_nosql) {
                    pew.restartProcess(LocalConstants.NOSQL_CLIENT, `${LocalConstants.NODE_V16_PATH} ${LocalConstants.NODEJS_SERVER_FILES_PATH}${LocalConstants.NOSQL_APP_PATH}`).then(() => {
                        res.end();
                    }).catch(() => {
                        return res.end();
                    });
                }
                //sql Not enabled
                else {
                    pew.killProcessByName(LocalConstants.NOSQL_CLIENT).then(() => {
                        res.end(LocalConstants.SAVE_OK);
                    }).catch(err => {
                        return res.end(err.message);
                    })
                }
                break;

            case LocalConstants.requests.READ_INTERFACE:
            case LocalConstants.requests.READ_PORTS:                
                if (firstRequestTime === 'undefined')
                    firstRequestTime = Date.now();
                else {
                    secondRequestTime = Date.now();
                    diff = secondRequestTime - firstRequestTime;
                    firstRequestTime = Date.now();
                }

                //Delay the action buy 10ms or 1500ms depending if the change of settings is from
                //the same request. In case of same request, we need to delay longer since the 1st request
                //might have not started yet and it could happen that the app is started twice
                delaytime = !diff || diff > 1100 ? 10 : 1500;
                setTimeout(function() {
                    //If one of the settings changed and the tlsServer is active, it has to be restarted, since parameter could have changed
                    pew.getPIDofProcess(LocalConstants.TLS_SERVER).then(() => {
                        //PID available -> restart it
                        pew.restartProcess(LocalConstants.TLS_SERVER, `${LocalConstants.NODE_V16_PATH} ${LocalConstants.NODEJS_SERVER_FILES_PATH}${LocalConstants.TLS_SERVER_PATH}`);
                    }).catch(() => {
                        //No PID available -> check if app has to be started
                        pew.getConfFile(LocalConstants.requests.READ_TLS).then(config => {
                            if (typeof config.enable_tlsserver !== LocalConstants.UNDEFINED) {
                                if (config.enable_tlsserver) {
                                    pew.startApp(`${LocalConstants.NODE_V16_PATH} ${LocalConstants.NODEJS_SERVER_FILES_PATH}${LocalConstants.TLS_SERVER_PATH}`);
                                }
                            }
                        }).catch(err => {
                            LocalConstants.DebugLog(err);
                        })
                    })

                    //If one of the settings changed and the sqlClient is active, it has to be restarted, since parameter could have changed
                    pew.getPIDofProcess(LocalConstants.SQL_CLIENT).then(() => {
                        //PID available -> restart it
                        pew.restartProcess(LocalConstants.SQL_CLIENT, `${LocalConstants.NODE_V16_PATH} ${LocalConstants.NODEJS_SERVER_FILES_PATH}${LocalConstants.SQL_APP_PATH}`);
                    }).catch(() => {
                        //No PID available -> check if app has to be started
                        pew.getConfFile(LocalConstants.requests.READ_SQL).then(config => {
                            if (typeof config.enable_sql !== LocalConstants.UNDEFINED) {
                                if (config.enable_sql) {
                                    pew.startApp(`${LocalConstants.NODE_V16_PATH} ${LocalConstants.NODEJS_SERVER_FILES_PATH}${LocalConstants.SQL_APP_PATH}`);
                                }
                            }
                        }).catch(err => {
                            LocalConstants.DebugLog(err);
                        })
                    })

                    //If one of the settings changed and the nosqlClient is active, it has to be restarted, since parameter could have changed
                    pew.getPIDofProcess(LocalConstants.NOSQL_CLIENT).then(() => {
                        //PID available -> restart it
                        pew.restartProcess(LocalConstants.NOSQL_CLIENT, `${LocalConstants.NODE_V16_PATH} ${LocalConstants.NODEJS_SERVER_FILES_PATH}${LocalConstants.NOSQL_APP_PATH}`);
                    }).catch(() => {
                        //No PID available -> check if app has to be started
                        pew.getConfFile(LocalConstants.requests.READ_NOSQL).then(config => {
                            if (typeof config.enable_nosql !== LocalConstants.UNDEFINED) {
                                if (config.enable_nosql) {
                                    pew.startApp(`${LocalConstants.NODE_V16_PATH} ${LocalConstants.NODEJS_SERVER_FILES_PATH}${LocalConstants.NOSQL_APP_PATH}`);
                                }
                            }
                        }).catch(err => {
                            LocalConstants.DebugLog(err);
                        })
                    })

                    //If one of the settings changed and the tlsClient is active, it has to be restarted, since parameter could have changed
                    pew.getPIDofProcess(LocalConstants.TLS_CLIENT).then(() => {
                        //PID available -> restart it
                        pew.restartProcess(LocalConstants.TLS_CLIENT, `${LocalConstants.NODE_V16_PATH} ${LocalConstants.NODEJS_SERVER_FILES_PATH}${LocalConstants.TLS_CLIENT_PATH}`).then(() => {
                            res.end(LocalConstants.SAVE_OK);
                        }).catch(() => {
                            return res.end(LocalConstants.SAVE_OK);
                        });
                    }).catch(() => {
                        //No PID available -> check if app has to be started
                        pew.getConfFile(LocalConstants.requests.READ_TLS).then(config => {
                            if (typeof config.enable_tlsclient !== LocalConstants.UNDEFINED) {
                                if (config.enable_tlsclient) {
                                    pew.startApp(`${LocalConstants.NODE_V16_PATH} ${LocalConstants.NODEJS_SERVER_FILES_PATH}${LocalConstants.TLS_CLIENT_PATH}`);
                                }
                            }
                        }).catch(err => {
                            LocalConstants.DebugLog(err);
                        })
                        res.end(LocalConstants.SAVE_OK);
                    })

                    //If one of the settings changed and the rs232Transparent is active, it has to be restarted, since parameter could have changed
                    pew.getPIDofProcess(LocalConstants.RS232_TRANSPARENT).then(() => {
                        //PID available -> restart it
                        pew.restartProcess(LocalConstants.RS232_TRANSPARENT, `${LocalConstants.NODE_V16_PATH} ${LocalConstants.NODEJS_SERVER_FILES_PATH}${LocalConstants.RS232TRANSPARENT_APP_PATH}`).then(() => {
                            res.end(LocalConstants.SAVE_OK);
                        }).catch(() => {
                            return res.end(LocalConstants.SAVE_OK);
                        });
                    }).catch(() => {
                        //No PID available -> check if app has to be started
                        pew.getConfFile(LocalConstants.requests.READ_INTERFACE).then(config => {
                            let rs232 = config.interface.filter(el => el.id === "/RS232");
                            if (rs232.length !== 0 && rs232[0].enabled === 1 && rs232[0].protocol === LocalConstants.PROTOCOLS.transparent) {
                                pew.startApp(`${LocalConstants.NODE_V16_PATH} ${LocalConstants.NODEJS_SERVER_FILES_PATH}${LocalConstants.RS232TRANSPARENT_APP_PATH}`);
                            }
                        }).catch(err => {
                            LocalConstants.DebugLog(err);
                        })
                        res.end(LocalConstants.SAVE_OK);
                    })

                    //If one of the settings changed and the rs485Transparent is active, it has to be restarted, since parameter could have changed
                    pew.getPIDofProcess(LocalConstants.RS485_TRANSPARENT).then(() => {
                        //PID available -> restart it
                        pew.restartProcess(LocalConstants.RS485_TRANSPARENT, `${LocalConstants.NODE_V16_PATH} ${LocalConstants.NODEJS_SERVER_FILES_PATH}${LocalConstants.RS485TRANSPARENT_APP_PATH}`).then(() => {
                            res.end(LocalConstants.SAVE_OK);
                        }).catch(() => {
                            return res.end(LocalConstants.SAVE_OK);
                        });
                    }).catch(() => {
                        //No PID available -> check if app has to be started
                        pew.getConfFile(LocalConstants.requests.READ_INTERFACE).then(config => {
                            let rs485 = config.interface.filter(el => el.id === "/RS485/RS232");
                            if (rs485.length !== 0 && rs485[0].enabled === 1 && rs485[0].protocol === LocalConstants.PROTOCOLS.transparent) {
                                pew.startApp(`${LocalConstants.NODE_V16_PATH} ${LocalConstants.NODEJS_SERVER_FILES_PATH}${LocalConstants.RS485TRANSPARENT_APP_PATH}`);
                            }
                        }).catch(err => {
                            LocalConstants.DebugLog(err);
                        })
                        res.end(LocalConstants.SAVE_OK);
                    })
                }, delaytime);
                break;

            case LocalConstants.requests.READ_TLS:
                if (typeof (req.body.enable_tlsclient) === LocalConstants.UNDEFINED) {
                    return res.end(LocalConstants.FAILED);
                }
                //tls Client enabled, start/restart the app
                req.body.enable_tlsclient ?
                    pew.restartProcess(LocalConstants.TLS_CLIENT, `${LocalConstants.NODE_V16_PATH} ${LocalConstants.NODEJS_SERVER_FILES_PATH}${LocalConstants.TLS_CLIENT_PATH}`) : pew.killProcessByName(LocalConstants.TLS_CLIENT);

                //tls Server enabled, start/restart the app
                if (req.body.enable_tlsserver) {
                    pew.restartProcess(LocalConstants.TLS_SERVER, `${LocalConstants.NODE_V16_PATH} ${LocalConstants.NODEJS_SERVER_FILES_PATH}${LocalConstants.TLS_SERVER_PATH}`).then(() => {
                        res.end();
                    }).catch(() => {
                        return res.end();
                    });
                }
                else {
                    pew.killProcessByName(LocalConstants.TLS_SERVER).then(() => {
                        res.end(LocalConstants.SAVE_OK);
                    }).catch(err => {
                        return res.end(err.message);
                    })
                }
                break;

            default:
                res.end(LocalConstants.SAVE_OK);
                break;
        }       

        //Settings saved, tell pew main application that settings have been changed
        pew.sendSIGToEmbeddedApp(LocalConstants.PEWMAIN);

    })
})

//Set up post for delete cert file
app.post(`${pew.Constants.requests.DELETE}*`, (req, res) =>
{
    //Don't allow requests from external
    if (!pew.isRequestFromLocalhost(req.ip))
    {
        pew.returnForbidden(req.ip, res);
        return;
    }

    var LocalConstants = pew.Constants;
    var uri = url.parse(req.url).pathname;
    var content = req.body;
    //Debug information
    LocalConstants.DebugLog(LocalConstants.DELETE_CERT_FILE_REQ);
    if (!pew.isUndefined(content))
    {
        switch (uri)
        {
            case LocalConstants.requests.DELETE_CERT:
                pew.deleteCertFile(content).then(() =>
                {
                    res.end(LocalConstants.DELETE_OK);
                }).catch(err =>
                {
                    return pew.returnWithFailHeaderAndMsg(`${LocalConstants.DELETE_FAILED}: ${err}`, LocalConstants.httpStatus.BAD_REQUEST, LocalConstants.DELETION_NOK, res);
                })
                break;

            case LocalConstants.requests.DELETE_LOG_FILE:
                pew.deleteLogFile(content).then(() =>
                {
                    res.end(LocalConstants.DELETE_OK);
                }).catch(err =>
                {
                    return pew.returnWithFailHeaderAndMsg(err, LocalConstants.httpStatus.BAD_REQUEST, err, res);
                })
                break;

            case LocalConstants.requests.DELETE_USER_WEB_FILES:
                pew.removeUserWebFiles().then(() => {
                    res.end(LocalConstants.DELETE_OK);
                }).catch(err => {
                    return pew.returnWithFailHeaderAndMsg(err.message, LocalConstants.httpStatus.BAD_REQUEST, err.message, res);
                });
                break;

            default:
                pew.returnWithFailHeaderAndMsg(false, LocalConstants.httpStatus.NOT_FOUND, LocalConstants.FAILED, res);
        }
    }
    else
    {
        pew.returnWithFailHeaderAndMsg(LocalConstants.DELETE_FAILED, LocalConstants.httpStatus.NOT_FOUND, LocalConstants.FAILED, res);
    }
})

//Set up post for write content to file (used for script file content)
app.post(pew.Constants.requests.WRITE, (req, res) => 
{
    //Don't allow requests from external
    if (!pew.isRequestFromLocalhost(req.ip))
    {
        pew.returnForbidden(req.ip, res);
        return;
    }

    var content = req.body;
    var filecontent = content.filecontent;

    //Debug information
    pew.Constants.DebugLog(`${pew.Constants.WRITE_TO_SCRIPT_REQ}: ${content.filename}`);

    if (typeof filecontent !== pew.Constants.UNDEFINED && filecontent !== pew.Constants.EMPTY_STRING)
    {
        try 
        {
            pew.writeContentToFile(content).then(() =>
            {
                if (!pew.rewriteConfFile(path.join(__dirname, pew.Constants.configFolder, pew.Constants.requests.READ_SCRIPT + pew.Constants.JSONFILE), pew.Constants.SCRIPT_UPLOAD_NAME, pew.Constants.SCRIPT_FILE_NAME))
                {
                    res.end();
                }
                else
                {
                    res.write(pew.Constants.WRITE_TO_SCRIPT_FAILED);
                    res.end();
                }
            }).catch(() =>
            {
                return pew.returnWithFailHeaderAndMsg(pew.Constants.WRITE_TO_SCRIPT_FAILED, pew.Constants.httpStatus.BAD_REQUEST, pew.Constants.FAILED, res);
            })
        }
        catch (ex)
        {
            pew.returnWithFailHeaderAndMsg(`${pew.Constants.WRITE_TO_SCRIPT_FAILED}: ${ex.message}`, false, ex.message, res);
        }
    }
    else 
    {
        //Empty file -> just response with no comment
        res.end();
    }
})

//Set up post for restoring settings with backup file
app.post(pew.Constants.requests.RESTORE, (req, res) => 
{
    //Don't allow requests from external
    if (!pew.isRequestFromLocalhost(req.ip))
    {
        pew.returnForbidden(req.ip, res);
        return;
    }

    var content = req.body;
    var filecontent = content.restorefile;
    var LocalConstants = pew.Constants;

    //Debug information
    LocalConstants.DebugLog(LocalConstants.SETTING_RESTORE_REQ);
    console.log(LocalConstants.SETTING_RESTORE_REQ);

    if (typeof filecontent !== LocalConstants.UNDEFINED && filecontent !== LocalConstants.EMPTY_STRING)
    {
        pew.restoreSettings(content).then(result =>
        {
            res.end(result);
        }).catch(err =>
        {
            pew.returnWithFailHeaderAndMsg(`${LocalConstants.RESTORE_FAILED}: ${err}`, pew.Constants.httpStatus.BAD_REQUEST, err, res);
        })
    }
    else 
    {
        pew.returnWithFailHeaderAndMsg(LocalConstants.RESTORE_FILE_EMPTY, pew.Constants.httpStatus.BAD_REQUEST, LocalConstants.FAILED, res);
    }
})

//Set up post for plc test
app.post(pew.Constants.requests.PLCTEST, (req, res) =>
{
    //Don't allow requests from external
    if (!pew.isRequestFromLocalhost(req.ip))
    {
        pew.returnForbidden(req.ip, res);
        return;
    }

    var content = req.body
    var LocalConstants = pew.Constants;

    pew.plcCommunicationTest(content).then(msg =>
    {
        res.end(msg);
    }).catch(err =>
    {
        pew.returnWithFailHeaderAndMsg(`${LocalConstants.PLC_TEST_FAIL}: ${err}`, LocalConstants.httpStatus.SERVER_ERR, err, res);
    })
 
})

//Set up post for set values
app.post(`${pew.Constants.requests.SET}*`, (req, res) =>
{
    let uri = url.parse(req.url).pathname;
    let content = req.body;

    switch (uri)
    {
        case pew.Constants.requests.SET_BIT:
            if (pew.Constants.isCustomized)
            {
                pew.setBit(content).then((result) =>
                {
                    res.end(result);
                }).catch(err =>
                {
                    res.end(err);
                })
            }
            else
            {
                pew.returnWithFailHeaderAndMsg(false, LocalConstants.httpStatus.FORBIDDEN, LocalConstants.NOT_ALLOWED, res);
            }
            break;

        case pew.Constants.requests.SET_PLC:
            if (pew.Constants.isCustomized)
            {
                pew.updatePLCProgram(content).then(result =>
                {
                    res.end(result);
                }).catch(err =>
                {
                    pew.returnWithFailHeaderAndMsg(false, LocalConstants.httpStatus.SERVER_ERR, err, res);
                });
            }
            else
            {
                return pew.returnWithFailHeaderAndMsg(false, LocalConstants.httpStatus.FORBIDDEN, LocalConstants.NOT_ALLOWED, res);
            }
            break;

        case pew.Constants.requests.SET_PLC_DEFAULT:
            if (pew.Constants.isCustomized)
            {
                pew.setPLCToDefault().then(result =>
                {
                    res.end(result);
                }).catch(err =>
                {
                    pew.returnWithFailHeaderAndMsg(false, LocalConstants.httpStatus.SERVER_ERR, err, res);
                });
            }
            else
            {
                return pew.returnWithFailHeaderAndMsg(false, LocalConstants.httpStatus.FORBIDDEN, LocalConstants.NOT_ALLOWED, res);
            }
            break;
    }
})

//Prepare upload with multer 
var storage = multer.diskStorage({
    destination: function (req, file, cb) 
    {
        let folderPath;
        switch (file.fieldname)
        {
            case pew.Constants.certFilename.cacert:
            case pew.Constants.certFilename.client:
            case pew.Constants.certFilename.key:
                folderPath = path.join(__dirname, pew.Constants.configSubfolders.mqtt);
                pew.createFolderIfNotExists(folderPath);
                pew.removeCertBe4Upload(file.originalname.substring(0, file.originalname.lastIndexOf('.')), folderPath);
                cb(null, folderPath);
                break;

            case pew.Constants.certFilename.cacert_ftps:
            case pew.Constants.certFilename.client_ftps:
            case pew.Constants.certFilename.key_ftps:
                folderPath = path.join(__dirname, pew.Constants.configSubfolders.ftpc);
                pew.createFolderIfNotExists(folderPath);
                pew.removeCertBe4Upload(file.originalname.substring(0, file.originalname.lastIndexOf('.')), folderPath);
                cb(null, folderPath);
                break;

            case pew.Constants.certFilename.cacert_https:
            case pew.Constants.certFilename.client_https:
            case pew.Constants.certFilename.key_https:
                folderPath = path.join(__dirname, pew.Constants.configSubfolders.http);
                pew.createFolderIfNotExists(folderPath);
                pew.removeCertBe4Upload(file.originalname.substring(0, file.originalname.lastIndexOf('.')), folderPath);
                cb(null, folderPath);
                break;

            case pew.Constants.certFilename.cacert_smtp:
            case pew.Constants.certFilename.client_smtp:
            case pew.Constants.certFilename.key_smtp:
                folderPath = path.join(__dirname, pew.Constants.configSubfolders.smtp);
                pew.createFolderIfNotExists(folderPath);
                pew.removeCertBe4Upload(file.originalname.substring(0, file.originalname.lastIndexOf('.')), folderPath);
                cb(null, folderPath);
                break;

            case pew.Constants.certFilename.slqclientca:
            case pew.Constants.certFilename.sqlclientcert:
            case pew.Constants.certFilename.sqlclientkey:
                folderPath = path.join(__dirname, pew.Constants.configSubfolders.sql);
                pew.createFolderIfNotExists(folderPath);
                pew.removeCertBe4Upload(file.originalname.substring(0, file.originalname.lastIndexOf('.')), folderPath);
                cb(null, folderPath);
                break;

            case pew.Constants.certFilename.noslqclientca:
            case pew.Constants.certFilename.nosqlclientcert:
            case pew.Constants.certFilename.nosqlclientkey:
                folderPath = path.join(__dirname, pew.Constants.configSubfolders.nosql);
                pew.createFolderIfNotExists(folderPath);
                pew.removeCertBe4Upload(file.originalname.substring(0, file.originalname.lastIndexOf('.')), folderPath);
                cb(null, folderPath);
                break;

            case pew.Constants.LOG_FORMAT_FILE_NAME:
                folderPath = path.join(__dirname, pew.Constants.configSubfolders.log);
                pew.createFolderIfNotExists(folderPath);
                cb(null, folderPath);
                break;

            case pew.Constants.SCRIPT_UPLOAD_NAME:
                folderPath = path.join(__dirname, pew.Constants.configSubfolders.script);
                pew.createFolderIfNotExists(folderPath);
                cb(null, folderPath);
                break;

            case pew.Constants.firmwareUpdateFiles.package:
            case pew.Constants.firmwareUpdateFiles.md5:
            case pew.Constants.USER_WEB_FILES:
                folderPath = path.join(__dirname, pew.Constants.tempFolder);
                pew.createFolderIfNotExists(folderPath);
                cb(null, folderPath);
                break;

            case pew.Constants.certFilename.cacert_tlsclient:
            case pew.Constants.certFilename.client_tlsclient:
            case pew.Constants.certFilename.key_tlsclient:
            case pew.Constants.certFilename.cacert_tlsserver:
            case pew.Constants.certFilename.cert_tlsserver:
            case pew.Constants.certFilename.key_tlsserver:
                folderPath = path.join(__dirname, pew.Constants.configSubfolders.tls);
                pew.createFolderIfNotExists(folderPath);
                pew.removeCertBe4Upload(file.fieldname, folderPath);
                cb(null, folderPath);
                break;

            default:
                cb(null, path.join(__dirname, pew.Constants.configFolder));
        }
    },
    filename: function (req, file, cb) 
    {
        var filename;
        switch (file.fieldname) 
        {
            //tls
            case pew.Constants.certFilename.cacert_tlsclient:
                filename = pew.Constants.certFilename.cacert_tlsclient + path.extname(file.originalname);
                break;

            case pew.Constants.certFilename.client_tlsclient:
                filename = pew.Constants.certFilename.client_tlsclient + path.extname(file.originalname);
                break;

            case pew.Constants.certFilename.key_tlsclient:
                filename = pew.Constants.certFilename.key_tlsclient + path.extname(file.originalname);
                break;

            case pew.Constants.certFilename.cacert_tlsserver:
                filename = pew.Constants.certFilename.cacert_tlsserver + path.extname(file.originalname);
                break;

            case pew.Constants.certFilename.cert_tlsserver:
                filename = pew.Constants.certFilename.cert_tlsserver + path.extname(file.originalname);
                break;

            case pew.Constants.certFilename.key_tlsserver:
                filename = pew.Constants.certFilename.key_tlsserver + path.extname(file.originalname);
                break;

            case pew.Constants.certFilename.cacert:
            case pew.Constants.certFilename.cacert_ftps:
            case pew.Constants.certFilename.cacert_smtp:
            case pew.Constants.certFilename.slqclientca:
                filename = pew.Constants.certFilename.cacert + path.extname(file.originalname);
                break;

            case pew.Constants.certFilename.client:
            case pew.Constants.certFilename.client_ftps:
            case pew.Constants.certFilename.client_smtp:
            case pew.Constants.certFilename.sqlclientcert:
                filename = pew.Constants.certFilename.client + path.extname(file.originalname);
                break;

            case pew.Constants.certFilename.key:
            case pew.Constants.certFilename.key_ftps:
            case pew.Constants.certFilename.key_smtp:
            case pew.Constants.certFilename.sqlclientkey:
                filename = pew.Constants.certFilename.key + path.extname(file.originalname);
                break;

            //all others
            case pew.Constants.LOG_FORMAT_FILE_NAME:
                filename = file.originalname;
                break;

            case pew.Constants.SCRIPT_UPLOAD_NAME:
                filename = pew.Constants.SCRIPT_FILE_NAME;
                break;

            case pew.Constants.firmwareUpdateFiles.package:
                filename = pew.Constants.firmwareFixedNames.package;
                break;

            case pew.Constants.firmwareUpdateFiles.md5:
                filename = pew.Constants.firmwareFixedNames.md5;
                break;

            case pew.Constants.USER_WEB_FILES:
                filename = pew.Constants.USER_WEB_FILE_NAME;
                break;
            
            default:
                filename = file.originalname;
        }

        cb(null, filename);
    }
})

var upload = multer({
    storage: storage,
    limits: { fileSize: pew.Constants.MAX_UPLOAD_SIZE },
    fileFilter: function (req, file, callback) {
        var fileExtension = path.extname(file.originalname).toUpperCase();
        //Debug information
        pew.Constants.DebugLog(`${pew.Constants.FILE_UPLOAD_REQ}: ${file.originalname} with fieldname: ${file.fieldname}`);

        switch (file.fieldname) 
        {
            //Check certificate file extension
            case pew.Constants.certFilename.cacert:
            case pew.Constants.certFilename.client:
            case pew.Constants.certFilename.cacert_ftps:
            case pew.Constants.certFilename.client_ftps:
            case pew.Constants.certFilename.cacert_https:
            case pew.Constants.certFilename.client_https:
            case pew.Constants.certFilename.cacert_smtp:
            case pew.Constants.certFilename.client_smtp:
            case pew.Constants.certFilename.cacert_tlsclient:
            case pew.Constants.certFilename.client_tlsclient:
            case pew.Constants.certFilename.cacert_tlsserver:
            case pew.Constants.certFilename.cert_tlsserver:
            case pew.Constants.certFilename.slqclientca:
            case pew.Constants.certFilename.sqlclientcert:
            case pew.Constants.certFilename.noslqclientca:
            case pew.Constants.certFilename.nosqlclientcert:
                if (fileExtension !== pew.Constants.certTypes.crt && fileExtension !== pew.Constants.certTypes.pem && fileExtension !== pew.Constants.certTypes.csr && fileExtension !== pew.Constants.certTypes.der && fileExtension !== pew.Constants.certTypes.cer) 
                {
                    //Wrong file type -> return an error
                    req.fileValidationError = pew.Constants.WRONG_FILE_FORMAT;
                    return callback(null, false, new Error(req.fileValidationError));
                }
                //No error -> return ok
                callback(null, true);
                break;
            //Check key file extension
            case pew.Constants.certFilename.key:
            case pew.Constants.certFilename.key_ftps:
            case pew.Constants.certFilename.key_https:
            case pew.Constants.certFilename.key_smtp:
            case pew.Constants.certFilename.key_tlsclient:
            case pew.Constants.certFilename.key_tlsserver:
            case pew.Constants.certFilename.sqlclientkey:
            case pew.Constants.certFilename.nosqlclientkey:
                if (fileExtension !== pew.Constants.certTypes.key && fileExtension != pew.Constants.certTypes.pem) 
                {
                    //Wrong key file type -> return an error
                    req.fileValidationError = pew.Constants.WRONG_FILE_FORMAT;
                    return callback(null, false, new Error(req.fileValidationError));
                }
                //No error -> return ok
                callback(null, true);
                break;
            //Log format file
            //Script file
            case pew.Constants.LOG_FORMAT_FILE_NAME:
            case pew.Constants.SCRIPT_UPLOAD_NAME:
                if (fileExtension !== pew.Constants.LOG_FILE_FORMAT) 
                {
                    req.fileValidationError = pew.Constants.WRONG_FILE_FORMAT;
                    return callback(null, false, new Error(req.fileValidationError));
                }
                //No error -> return ok
                callback(null, true);
                break;
            //firmware update files
            case pew.Constants.firmwareUpdateFiles.package:
                if (fileExtension !== pew.Constants.firmwareFileTypes.package)
                {
                    req.fileValidationError = pew.Constants.WRONG_FILE_FORMAT;
                    return callback(null, false, new Error(req.fileValidationError));
                }
                //No error -> return ok
                callback(null, true);
                break;
            case pew.Constants.firmwareUpdateFiles.md5:
                if (fileExtension !== pew.Constants.firmwareFileTypes.md5)
                {
                    req.fileValidationError = pew.Constants.WRONG_FILE_FORMAT;
                    return callback(null, false, new Error(req.fileValidationError));
                }
                //No error -> return ok
                callback(null, true);
                break;

            //User web files as zip file. File type will be checked in frontend
            case pew.Constants.USER_WEB_FILES:
                callback(null, true);
                break;
                
            default:
                //Debug information
                pew.Constants.DebugLog(pew.Constants.FILE_UPLOAD_FAIL);
                return callback(null, false, new Error(pew.Constants.FAILED))
        }


        switch (file.fieldname) {
            case pew.Constants.firmwareUpdateFiles.package:
            case pew.Constants.firmwareUpdateFiles.md5:
            case pew.Constants.USER_WEB_FILES:
                break;

            default:
                pew.writeCertFile(file);
        }
    }
})

//Allowed fieldnames for upload
var certUpload = upload.fields(
    [
        { name: pew.Constants.certFilename.cacert },
        { name: pew.Constants.certFilename.client },
        { name: pew.Constants.certFilename.key },
        { name: pew.Constants.certFilename.cacert_ftps },
        { name: pew.Constants.certFilename.client_ftps },
        { name: pew.Constants.certFilename.key_ftps },
        { name: pew.Constants.certFilename.cacert_https },
        { name: pew.Constants.certFilename.client_https },
        { name: pew.Constants.certFilename.key_https },
        { name: pew.Constants.certFilename.cacert_smtp },
        { name: pew.Constants.certFilename.client_smtp },
        { name: pew.Constants.certFilename.key_smtp },
        { name: pew.Constants.LOG_FORMAT_FILE_NAME },
        { name: pew.Constants.SCRIPT_UPLOAD_NAME },
        { name: pew.Constants.firmwareUpdateFiles.package },
        { name: pew.Constants.firmwareUpdateFiles.md5 },
        { name: pew.Constants.USER_WEB_FILES },
        { name: pew.Constants.certFilename.cacert_tlsclient },
        { name: pew.Constants.certFilename.client_tlsclient },
        { name: pew.Constants.certFilename.key_tlsclient },
        { name: pew.Constants.certFilename.cacert_tlsserver },
        { name: pew.Constants.certFilename.cert_tlsserver },
        { name: pew.Constants.certFilename.key_tlsserver },
        { name: pew.Constants.certFilename.slqclientca },
        { name: pew.Constants.certFilename.sqlclientcert },
        { name: pew.Constants.certFilename.sqlclientkey },
        { name: pew.Constants.certFilename.noslqclientca },
        { name: pew.Constants.certFilename.nosqlclientcert },
        { name: pew.Constants.certFilename.nosqlclientkey }
    ]);
//Set upload with multer
app.post(pew.Constants.requests.UPLOAD, certUpload, function (req, res) 
{
    //File validation ok?
    if (req.fileValidationError)
    {
        return res.end(req.fileValidationError);
    }

    //Check if any files has been selected
    if (Object.keys(req.files).length === 0)
    {
        return res.end(pew.Constants.NO_FILE_SELECTED);
    }

    //Check if uploaded files are certificates for tlsClient or tlsServer. If so restart the services
    let bIsTLSClientCert = false, bIsTLSServerCert = false;
    if (Object.keys(req.files).length > 0) {
        for (let name of Object.keys(req.files)) {
            if (name === pew.Constants.certFilename.cacert_tlsclient || name === pew.Constants.certFilename.client_tlsclient || name === pew.Constants.certFilename.key_tlsclient) {
                bIsTLSClientCert = true;
            }

            if (name === pew.Constants.certFilename.cacert_tlsserver || name === pew.Constants.certFilename.cert_tlsserver || name === pew.Constants.certFilename.key_tlsserver) {
                bIsTLSServerCert = true;
            }
        }
    }

    //If one of the settings changed and the tlsClient is active, it has to be restarted, since parameter could have changed
    if (bIsTLSClientCert) {
        pew.getPIDofProcess(pew.Constants.TLS_CLIENT).then(() => {
            //PID available -> restart it
            pew.restartProcess(pew.Constants.TLS_CLIENT, `${pew.Constants.NODE_V16_PATH} ${pew.Constants.NODEJS_SERVER_FILES_PATH}${pew.Constants.TLS_CLIENT_PATH}`);
        }).catch(() => {
            //No PID available -> do nothing
        })
    }

    if (bIsTLSServerCert) {
        pew.getPIDofProcess(pew.Constants.TLS_SERVER).then(() => {
            //PID available -> restart it
            pew.restartProcess(pew.Constants.TLS_SERVER, `${pew.Constants.NODE_V16_PATH} ${pew.Constants.NODEJS_SERVER_FILES_PATH}${pew.Constants.TLS_SERVER_PATH}`);
        }).catch(() => {
            //No PID available -> do nothing
        })
    }

    //Check if the uploaded file(s) are firmware update files
    let bIsUpdateFiles = false;
    let bIsUserWebFiles = false;
    for (var i = 0; i < Object.keys(req.files).length; i++)
    {
        if (Object.keys(req.files)[i] === pew.Constants.USER_WEB_FILES) {
            bIsUserWebFiles = true;
            break;
        }

        if (Object.keys(req.files)[i] === pew.Constants.firmwareUpdateFiles.package || Object.keys(req.files)[i] === pew.Constants.firmwareUpdateFiles.md5)
        {
            bIsUpdateFiles = true;
            break;
        }
    }

    //If not return success
    if (!bIsUpdateFiles && !bIsUserWebFiles)
    {
        return res.end(pew.Constants.UPLOAD_OK);
    }

    if (bIsUpdateFiles) {
        //Firmware update files -> do more things in background
        pew.updateFirmware().then(() => {
            res.end(pew.Constants.UPLOAD_OK);
        }).catch(err => {
            pew.returnWithFailHeaderAndMsg(false, pew.Constants.httpStatus.BAD_REQUEST, err, res);
        })
    }

    if (bIsUserWebFiles) {
        pew.extractUserWebFiles().then(() => {
            res.end(pew.Constants.UPLOAD_OK);
        }).catch(() => {
            res.end();
        })
    }
    
})

//Start service
app.listen(pew.Constants.port, 'localhost', function ()
{
    //Debug information
    console.log(`${pew.Constants.NODE_APP_START}: ${pew.Constants.port.toString()}`)
    pew.Constants.DebugLog(`${pew.Constants.NODE_APP_START}: ${pew.Constants.port.toString()}`);
})